const { app, BrowserWindow, ipcMain, dialog } = require('electron');
const path = require('path');
const { spawn } = require('child_process');
const fs = require('fs');

let mainWindow;
let splashWindow;
let nextServer;
const isDev = process.env.NODE_ENV === 'development';
const port = process.env.PORT || 3000;

function createSplashWindow() {
  splashWindow = new BrowserWindow({
    width: 500,
    height: 400,
    frame: false,
    transparent: true,
    alwaysOnTop: true,
    resizable: false,
    webPreferences: {
      nodeIntegration: false,
      contextIsolation: true,
    },
  });

  splashWindow.loadFile(path.join(__dirname, 'splash.html'));
  splashWindow.center();
}

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    minWidth: 1024,
    minHeight: 768,
    webPreferences: {
      preload: path.join(__dirname, 'preload.js'),
      nodeIntegration: false,
      contextIsolation: true,
      webSecurity: true,
    },
    backgroundColor: '#0a0a0a',
    show: false, // Don't show until app is ready
  });

  // Show window when ready and close splash
  mainWindow.once('ready-to-show', () => {
    // Signal splash screen that app is ready
    if (splashWindow && !splashWindow.isDestroyed()) {
      splashWindow.webContents.executeJavaScript(`
        window.dispatchEvent(new Event('app-ready'));
      `);
    }

    // Wait for splash fade out animation
    setTimeout(() => {
      if (splashWindow && !splashWindow.isDestroyed()) {
        splashWindow.close();
        splashWindow = null;
      }
      mainWindow.show();
    }, 800);
  });

  // Add error handler for load failures
  mainWindow.webContents.on('did-fail-load', (event, errorCode, errorDescription) => {
    console.error('Failed to load:', errorCode, errorDescription);
  });

  // Log user data path for debugging
  console.log('User data path:', app.getPath('userData'));

  // Inject default settings on first load
  mainWindow.webContents.on('did-finish-load', () => {
    const configPath = isDev
      ? path.join(__dirname, 'default-config.json')
      : path.join(process.resourcesPath, 'electron', 'default-config.json');

    if (fs.existsSync(configPath)) {
      const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

      // Inject settings into localStorage if not already set
      mainWindow.webContents.executeJavaScript(`
        (function() {
          const storageKey = 'storyboard-storage';
          const existing = localStorage.getItem(storageKey);

          if (!existing) {
            // No existing settings, use defaults
            const defaultState = {
              state: {
                settings: ${JSON.stringify(config.settings)},
                storyboards: [],
                currentStoryboard: null
              },
              version: 0
            };
            localStorage.setItem(storageKey, JSON.stringify(defaultState));
            console.log('Injected default settings for Electron app');
          } else {
            // Check if settings exist and update if needed
            try {
              const state = JSON.parse(existing);
              if (!state.state || !state.state.settings || !state.state.settings.localSDEnabled) {
                // Update settings to enable local SD
                if (!state.state) state.state = {};
                state.state.settings = ${JSON.stringify(config.settings)};
                localStorage.setItem(storageKey, JSON.stringify(state));
                console.log('Updated settings to enable local SD');
              }
            } catch (e) {
              console.error('Error updating settings:', e);
            }
          }
        })();
      `);
    }
  });

  // Load the app
  const loadApp = async () => {
    try {
      if (isDev) {
        // Development mode - connect to existing dev server
        await mainWindow.loadURL(`http://localhost:${port}`);
        // mainWindow.webContents.openDevTools();
      } else {
        // Production mode - start Next.js standalone server
        const nextPath = path.join(process.resourcesPath, 'app');
        const serverPath = path.join(nextPath, 'server.js');

        // Check if server.js exists
        if (!fs.existsSync(serverPath)) {
          console.error('Server file not found:', serverPath);
          dialog.showErrorBox(
            'Application Error',
            'Server files not found. Please reinstall the application.'
          );
          app.quit();
          return;
        }

        // Start the Next.js standalone server
        // Use Electron's own executable as Node.js with ELECTRON_RUN_AS_NODE=1
        const electronPath = process.execPath;

        console.log('Starting Next.js server...');
        console.log('Electron path:', electronPath);
        console.log('Server path:', serverPath);
        console.log('Working directory:', nextPath);

        // Use Electron as Node.js by setting ELECTRON_RUN_AS_NODE
        nextServer = spawn(electronPath, [serverPath], {
          cwd: nextPath,
          env: {
            ...process.env,
            PORT: port.toString(),
            NODE_ENV: 'production',
            ELECTRON_RUN_AS_NODE: '1', // Makes Electron behave like Node.js
          },
          shell: false, // Don't use shell when using ELECTRON_RUN_AS_NODE
        });

        let serverReady = false;

        nextServer.stdout.on('data', (data) => {
          const message = data.toString();
          console.log(`Next.js: ${message}`);
          if (message.includes('Ready') || message.includes('started')) {
            serverReady = true;
          }
        });

        nextServer.stderr.on('data', (data) => {
          console.error(`Next.js Error: ${data}`);
        });

        nextServer.on('error', (error) => {
          console.error('Failed to start Next.js server:', error);
          dialog.showErrorBox(
            'Server Error',
            `Failed to start Next.js server: ${error.message}`
          );
        });

        nextServer.on('exit', (code, signal) => {
          console.log(`Next.js server exited with code ${code} and signal ${signal}`);
        });

        // Wait for server to be ready (with timeout)
        const maxWaitTime = 10000; // 10 seconds
        const startTime = Date.now();
        while (!serverReady && (Date.now() - startTime) < maxWaitTime) {
          await new Promise((resolve) => setTimeout(resolve, 500));
        }

        if (!serverReady) {
          console.log('Server not ready after timeout, trying to connect anyway...');
        } else {
          console.log('Server ready, waiting 2 more seconds for stability...');
          await new Promise((resolve) => setTimeout(resolve, 2000));
        }

        console.log('Loading URL: http://localhost:' + port);
        await mainWindow.loadURL(`http://localhost:${port}`);

        // Dev tools disabled for production
        // mainWindow.webContents.openDevTools();
      }
    } catch (error) {
      console.error('Failed to load app:', error);
      dialog.showErrorBox(
        'Application Error',
        `Failed to start the application: ${error.message}`
      );
    }
  };

  loadApp();

  // Handle window close event
  mainWindow.on('close', (event) => {
    // Cleanup Next.js server when window closes
    if (nextServer && !nextServer.killed) {
      console.log('Cleaning up Next.js server on window close...');
      if (process.platform === 'win32') {
        try {
          spawn('taskkill', ['/pid', nextServer.pid, '/f', '/t'], { shell: true });
        } catch (error) {
          console.error('Failed to kill Next.js server on close:', error);
        }
      } else {
        nextServer.kill('SIGTERM');
      }
    }
  });

  mainWindow.on('closed', () => {
    mainWindow = null;
  });
}

// App lifecycle
app.whenReady().then(() => {
  // Show splash screen first
  createSplashWindow();

  // Then create main window (hidden initially)
  createWindow();

  app.on('activate', () => {
    if (BrowserWindow.getAllWindows().length === 0) {
      createSplashWindow();
      createWindow();
    }
  });
});

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('before-quit', () => {
  // Kill the Next.js server when quitting
  if (nextServer) {
    // On Windows, we need to kill the entire process tree
    if (process.platform === 'win32') {
      try {
        // Use taskkill to kill the process tree
        spawn('taskkill', ['/pid', nextServer.pid, '/f', '/t'], { shell: true });
      } catch (error) {
        console.error('Failed to kill Next.js server:', error);
      }
    } else {
      nextServer.kill('SIGTERM');
    }
  }
});

// Handle app quit event
app.on('quit', () => {
  // Ensure server is killed on quit
  if (nextServer) {
    try {
      if (process.platform === 'win32') {
        spawn('taskkill', ['/pid', nextServer.pid, '/f', '/t'], { shell: true });
      } else {
        nextServer.kill('SIGKILL');
      }
    } catch (error) {
      console.error('Error during cleanup:', error);
    }
  }
});

// IPC handlers for file operations and other native features
ipcMain.handle('select-directory', async () => {
  const result = await dialog.showOpenDialog(mainWindow, {
    properties: ['openDirectory'],
  });
  return result.filePaths[0];
});

ipcMain.handle('save-file', async (event, options) => {
  const result = await dialog.showSaveDialog(mainWindow, options);
  return result.filePath;
});

// Handle app version
ipcMain.handle('get-app-version', () => {
  return app.getVersion();
});

// Handle export directory selection
ipcMain.handle('select-export-directory', async () => {
  const result = await dialog.showOpenDialog(mainWindow, {
    properties: ['openDirectory', 'createDirectory'],
    title: 'Select Export Location',
  });
  return result.filePaths[0];
});

// Handle saving image file directly
ipcMain.handle('save-image-file', async (event, { directory, filename, buffer }) => {
  try {
    const { writeFile } = require('fs/promises');
    const filepath = path.join(directory, filename);
    await writeFile(filepath, Buffer.from(buffer));
    return { success: true, filepath };
  } catch (error) {
    console.error('Failed to save image file:', error);
    return { success: false, error: error.message };
  }
});
